#!/bin/sh

# hard code the variables
#snapshotPart="/dev/mapper/Pixiebob-data_backup"
#mdPart="/dev/mapper/Pixiebob-data_md"
VG_NAME="Pixiebob"
#old_snap_le=32
#new_snap_le=320
#md_le=32

# F_ddd_ini_get
# Get the value of a key in an INI file
# Usage: Catch the value using an variable.
# e.g. VAR=`F_ini_get "file" "section" "key"`
#
# param: full path to an INI file
# param: section name excluding '[' ']'
# param: key name
F_ddd_ini_get()
{
    INI_F="$1"
    INI_SECTION="$2"
    INI_KEY="$3"

    if [ ! -f "$INI_F" ] ; then
        #echo "Error: ${INI_F} not found."  # not show error msg here, as the caller of F_ini_get always simply judge if the output of the function is null; here change this error with a different return code: 2
        return 2
    fi

    RANGE_START="/^\[${INI_SECTION}\]/"
    RANGE_END="/^\[.*\]/"

    MY_SEDCMD_1="${RANGE_START},${RANGE_END}p"
    MY_SEDCMD_2="/^[\t ]*${INI_KEY} = /p"

    _R=`sed -n -e "${MY_SEDCMD_1}" ${INI_F} | sed -n -e "${MY_SEDCMD_2}" | awk -F' = ' '{ print $2 }'`
    _R=`echo $_R`
    printf "%s" "$_R"
    if [ -z "$_R" ] ; then
        return 1
    else
        return 0
    fi
}

# F_ddd_ini_set
# Set the value of a key in an INI file
# It uses a temp file.
# e.g. F_ddd_ini_set "file" "section" "key" "value"
#
# param: full path to an INI file.
# param: section name excluding '[' ']'
# param: key name
# param: value
#
# print: error
#
F_ddd_ini_set() {
    INI_F="$1"
    INI_SECTION="$2"
    INI_KEY="$3"
    INI_VALUE="$4"
    TMP_F=/tmp/temp$$

    if [ ! -f "$INI_F" ] ; then
        echo "Error: ${INI_F} not found."
    return 1
    fi

    RANGE_START="/\[${INI_SECTION}\]/"
    RANGE_END="/\\[.*\]/"

    MY_SEDCMD="${RANGE_START},${RANGE_END}s?^[ \t]*${INI_KEY} = .*?${INI_KEY} = ${INI_VALUE}?"
    sed -e "${MY_SEDCMD}" ${INI_F} > ${TMP_F}
    cp -f ${TMP_F} ${INI_F}
    rm -f ${TMP_F}
}

# Description:
#    Get service status
# Arguments:
#    service name
# Return:
#    1: running
#    0: stopped
#    2: not found
get_status()
{
    local service=$1
    systemctl -h >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        # use systemctl
        systemctl list-units --type=service | grep "^$service" >/dev/null 2>&1
        if [ $? -eq 0 ]; then	# service exists
            systemctl is-active $service | grep "^active" >/dev/null 2>&1
            if [ $? -eq 0 ]; then
            return 1
            else
            return 0
            fi
        else	# service doesn't exist
            return 2
        fi
    else
        # upstart
        status $service 2>&1 | grep "Unknown job" >/dev/null 2>&1
        if [ $? -ne 0 ]; then	# service exists
            status $service | grep "stop" >/dev/null 2>&1
            if [ $? -ne 0 ]; then
            return 1
            else
            return 0
            fi
        else	# service doesn't exist
            return 2
        fi
    fi
}

# Descritpion:
#    Set service status
# Arguments:
#    service name
#    status: start|stop|restart
# Return:
#    0: success
#    other: fail
set_status()
{
    local service=$1
    local status=$2

    systemctl -h >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        # use systemctl
        systemctl $status $service >/dev/null 2>&1
        return $?
    else
        # upstart
        $status $service >/dev/null 2>&1
        return $?
    fi
}

# Description:
# chown home dir with uid:gid from specified passwd
# Arguments:
#    passwd file path
#    username
#    root path for homedir
# Return:
#    0: success
#    other: fail
reload_homedir()
{
    local passwd_file="$1"
    local username="$2"
    local root_path="$3"
    local homedir=`grep "^$username:" $passwd_file |  awk -F':' '{print $6}'`
    if [ "$homedir" = "" ] || [ "$homedir" = "/" ]; then
        return 0
    fi

    local uid=`grep "^$username:" $passwd_file |  awk -F':' '{print $3}'`
    local gid=`grep "^$username:" $passwd_file |  awk -F':' '{print $4}'`
    if [ "$uid" = "" ] || [ "$gid" = "" ]; then
        return 1
    fi

    chown -R $uid:$gid $root_path$homedir
    return $?
}

#Description: 
# Product callback function which will be executed before apply 
# patch/upgrade package to check if current system meets the 
# requirements of product.
#
#Arguments: 
# None
#
#Side effects:
# None
#
#Returns:
# 0 for success, others if failed.
F_product_pre_check()
{
    F_debug_log "In product pre-check."

    F_debug_log "Check files: Config/DB"
    PRODUCT_CONFIG=${UNPACK_DIR}/binaries/updateConfig.sh
    PRODUCT_DB=${UNPACK_DIR}/binaries/updateDb.sh

    if [ ! -e $PRODUCT_CONFIG ]; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Unable to locate the file \"updateConfig.sh\" in the package. The upgrade package should contain this file. ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FILE_MISSING
        return 1
    fi

    if [ ! -e $PRODUCT_DB ]; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Unable to locate the file \"updateDb.sh\" in the package. The upgrade package should contain this file. ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FILE_MISSING
        return 1
    fi
    F_debug_log "Check functions: Config/DB"
    source ${PRODUCT_CONFIG}
    source ${PRODUCT_DB}
    if ! F_function_exist F_product_config_pre_check; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_config_pre_check\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_config_pre_check\". File \"updateConfig.sh\" should define function \"F_product_config_pre_check\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_config_pre_apply; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_config_pre_apply\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_config_pre_apply\". File \"updateConfig.sh\" should define function \"F_product_config_pre_apply\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_config_post_apply; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_config_post_apply\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_config_post_apply\". File \"updateConfig.sh\" should define function \"F_product_config_post_apply\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_config_pre_apply_recovery; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_config_pre_apply_recovery\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_config_pre_apply_recovery\". File \"updateConfig.sh\" should define function \"F_product_config_pre_apply_recovery\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_config_post_apply_recovery; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_config_post_apply_recovery\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_config_post_apply_recovery\". File \"updateConfig.sh\" should define function \"F_product_config_post_apply_recovery\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi

    if ! F_function_exist F_product_db_pre_check; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_db_pre_check\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_db_pre_check\". File \"updateDb.sh\" should define function \"F_product_db_pre_check\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_db_pre_apply; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_db_pre_apply\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_db_pre_apply\". File \"updateDb.sh\" should define function \"F_product_db_pre_apply\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_db_post_apply; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_db_post_apply\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_db_post_apply\". File \"updateDb.sh\" should define function \"F_product_db_post_apply\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_db_pre_apply_recovery; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_db_pre_apply_recovery\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_db_pre_apply_recovery\". File \"updateDb.sh\" should define function \"F_product_db_pre_apply_recovery\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi
    if ! F_function_exist F_product_db_post_apply_recovery; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Function \"F_product_db_post_apply_recovery\" doesn't exist."
        F_prompt_log "Unable to find function \"F_product_db_post_apply_recovery\". File \"updateDb.sh\" should define function \"F_product_db_post_apply_recovery\". ${ERRMSG_REGEN_PKG}."
        ERROR_NUMBER=$ERR_FUNC_MISSING
        return 1
    fi

    # check product version
    F_debug_log "Check product version."
    PACKAGE_CONFIG=/var/tmp/config.ini
    is_skip_version_check=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "SkipVersionCheck"`
    if [ "${is_skip_version_check}" = "1" ]; then
        F_debug_log "*** (only for test) Skip check product version ***"
    else
        language_o="$LANG"
        if [ -f /etc/profile.d/product.sh ]; then	# product info is defined in enviroment variable
            . /etc/profile.d/product.sh
            product_name_o="$PRODUCT_SHORT_NAME"
            product_version_o="${PRODUCT_VERSION}.${PRODUCT_SERVICE_PACK_NUM}"
            if [ "${PRODUCT_HOTFIX_VERSION}" = "" ]; then
                product_build_o="${PRODUCT_BUILD_NUM}"
            else
                product_build_o="${PRODUCT_HOTFIX_VERSION}"
            fi
        fi
        if [ "${language_o}" = "" ] || [ "${product_name_o}" = "" ] || [ "${product_version_o}" = "" ] || [ "${product_build_o}" = "" ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot get current product information."
            return 1
        fi
        F_debug_log "language_o: $language_o, product_name_o: $product_name_o, product_version_o: $product_version_o, product_build_o: $product_build_o"
        language_n=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "Language"`
        product_name_n=`F_ini_get ${PACKAGE_CONFIG} "General" "ProductName"`
        product_version_n=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "AppVersion"`
        product_build_n=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "AppBuild"`
        product_support_version_1=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "SupportAppVersion1"`
        product_support_build_1=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "SupportBuild1"`
        product_support_version_2=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "SupportAppVersion2"`
        product_support_build_2=`F_ini_get ${PACKAGE_CONFIG} "Upgrade" "SupportBuild2"`

        F_debug_log "language_n: $language_n, product_name_n: $product_name_n, product_version_n: $product_version_n, product_build_n: $product_build_n"
        F_debug_log "product_support_version_1: $product_support_version_1, product_support_build_1: $product_support_build_1, product_support_version_2: $product_support_version_2, product_support_build_2: $product_support_build_2"
        if [ "${language_n}" = "" ] || [ "${product_name_n}" = "" ] || [ "${product_version_n}" = "" ] || [ "${product_build_n}" = "" ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot get upgrade information."
            return 1
        fi
        if [ "$product_support_version_1" = "" ] && [ "$product_support_version_2" = "" ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot get support app version"
            return 1
        fi
        if [ "$product_support_build_1" = "" ] && [ "$product_support_build_2" = "" ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot get support app build"
            return 1
        fi

        if [ "$product_support_version_1" != "" ] &&  [ "$product_support_build_1" = "" ]; then
            F_error_log "Cannot get paired support (1) app version/build."
            return 1
        else
            F_debug_log "Support Version 1: $product_support_version_1 $product_support_build_1"
        fi

        if [ "$product_support_version_2" != "" ] && [ "$product_support_build_2" = "" ]; then
            F_error_log "Cannot get paired support (2) app version/build."
            return 1
        else
            F_debug_log "Support Version 2: $product_support_version_2 $product_support_build_2"
        fi

        if [ "${product_name_o}" != "${product_name_n}" ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot upgrade from ${product_name_o} to ${product_name_n}."
            return 1
        fi

        if [ "${language_o}" != "${language_n}" ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot upgrade from ${language_o} to ${language_n}."
            return 1
        fi

        if [ "${product_version_o}" = "${product_support_version_1}" -a ${product_build_o} -ge ${product_support_build_1} ]; then
            F_debug_log "Match supported version 1"
        elif [ "${product_version_o}" = "${product_support_version_2}" -a ${product_build_o} -ge ${product_support_build_2} ]; then
            F_debug_log "Match supported version 2"
        elif [ "${product_version_n}" = "${product_version_o}" -a ${product_build_n} -gt ${product_build_o} ]; then
            F_debug_log "Match. Product version is the same."
        else
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Cannot upgrade from ${product_version_o}.${product_build_o} to ${product_version_n}.${product_build_n}"
            return 1
        fi

        F_debug_log "Product version check success. Will upgrade from ${product_version_o}.${product_build_o} to ${product_version_n}.${product_build_n}"
    fi

    # Check avaiable space
    backup_dst=`F_ini_get ${PACKAGE_CONFIG} "Partition" "DataBackupMountPoint"`
    if [ "${backup_dst}" = "" ]; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Config DataBackupMountPoint in Partition secion is not set."
        return 1
    fi

    if [ ! -e "$backup_dst/ddd_backup" ];then
        F_debug_log "Create backup folder: $backup_dst/ddd_backup"
        /usr/bin/mkdir -p $backup_dst/ddd_backup/
        if [ $? -ne 0 ]; then
            F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Failed to prepare backup folder $backup_dst/ddd_backup"
            return 1
        fi
    fi
    avaiable_backup_dst_space=`df $backup_dst | grep -Ew "$backup_dst" | awk -F' ' '{print \$4}'`
    DDDDataBackupFile=$backup_dst/ddd_backup/ddd_upgrade_backup_data_${product_version_o}_${product_build_o}.tgz
    DDDConfigBackupFile=$backup_dst/ddd_backup/ddd_upgrade_backup_config_${product_version_o}_${product_build_o}.tar
    DDDDBBackupFile=$backup_dst/ddd_backup/ddd_upgrade_backup_db_${product_version_o}_${product_build_o}.tgz

    F_debug_log "Avaiable space of $backup_dst is $avaiable_backup_dst_space"

    F_product_config_pre_check || {
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Pre-Check of Product config failed."
        F_prompt_log "Product defined pre-check function \"F_product_config_pre_check\" execute failed. ${ERRMSG_ASK_HELP}."
        ERROR_NUMBER=${ERR_PRODUCT_PRE_CHECK}
        return 1
    } && {
        F_prompt_log "Pre-Check of product config succeeded."
    }
    F_product_db_pre_check || {
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Pre-Check of Product config failed."
        F_prompt_log "Product defined pre-check function \"F_product_db_pre_check\" execute failed. ${ERRMSG_ASK_HELP}."
        ERROR_NUMBER=${ERR_PRODUCT_PRE_CHECK}
        return 1
    } && {
        F_prompt_log "Pre-Check of product db succeeded."
    }

    return 0
}

#Description: 
# Product callback function which will be executed before apply upgrade.
#
#Arguments: 
# None
#
#Side effects:
# None
#
#Returns: 
# 0 if successful, others if failed.
F_product_pre_apply()
{
    F_debug_log "In product pre-apply."
    # This is optional, if there is no action need to do just leave it empty.
    F_add_recovery_func "F_product_pre_apply_recovery"

    db_service=postgresql-9.3.service
    rpm -q postgresql96
    flag_upgrade_to_pgsql96=$?
    if [ ${flag_upgrade_to_pgsql96} -eq 0 ]; then
        db_service=postgresql-9.6.service
    fi

    F_debug_log "Database service:${db_service}"

    for serv in ${db_service} pixiebob-agent pixiebob-portal pixiebob-resource pixiebob-scheduler pixiebob-log-receiver; do
        if [ $? -eq 0 ]; then # stopped
            F_debug_log "Stop $serv."
            set_status $serv stop
        else
            F_debug_log "$serv already stop, skip stop."
        fi
    done

    F_debug_log "Move LDAP cert folder"
    # Move LDAP folder
    old_ldap_path='/var/local_data/ldap_cert'
    new_ldap_path='/etc/pki/Pixiebob'
    ldap_moved=0
    if [ -d "${old_ldap_path}" ]; then
        mv -f "${old_ldap_path}" "${new_ldap_path}"
        ldap_moved=1
    fi

    # Backup DB
    # Backup config

    local noprompt=""
    lvcreate --version | grep "LVM version:" | grep "RHEL6" >/dev/null 2>&1
    if [ $? -ne 0 ]; then
        noprompt="--yes"
    fi

    #old_le=`F_get_lvsize $snapshotPart`
    #if [ -n "${old_le}" ] && [ ${old_le} -eq ${old_snap_le} ]; then
    #    F_prompt_log "Remove old snapshot lvm."
    #    # Drop old snapshot lvm if it = 1GB
    #    F_lv_remove $snapshotPart || {
    #        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Can't remove old snapshot lvm $snapshotPart."
    #        F_prompt_log "Unable to remove old snapshot lvm: \"$snapshotPart\". ${ERRMSG_ASK_HELP}."
    #        return 1
    #    }
    #fi
    #if [ ! -n "${old_le}" ] || ( [ -n "${old_le}" ] && [ ${old_le} -eq ${old_snap_le} ] ); then
    #    # Create the new snapshot lvm to 10GB
    #    F_prompt_log "Create new snapshot lvm."
    #    lvcreate --extents ${new_snap_le} --name ${snapshotPart##*$VG_NAME-} $VG_NAME ${noprompt} > /dev/null 2>&1 || {
    #        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Can't create new snapshot lvm $snapshotPart."
    #        F_prompt_log "Unable to create new snapshot lvm: \"$snapshotPart\". ${ERRMSG_ASK_HELP}."
    #        return 1
    #    }
    #fi

    #lvdisplay -c ${mdPart} >/dev/null 2>&1
    #if [ $? -ne 0 ]; then
    #    F_prompt_log "Create meta data lvm."
    #    # create the meta data lvm for HA if it doesn't exist
    #    lvcreate --extents ${md_le} --name=${mdPart##*$VG_NAME-} $VG_NAME ${noprompt} > /dev/null 2>&1 || {
    #        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Can't create meta data lvm $mdPart."
    #        F_prompt_log "Unable to create meta data lvm: \"$mdPart\". ${ERRMSG_ASK_HELP}."
    #        return 1
    #    }
    #fi

    F_debug_log "Mount /proc"
    umount -f ${SYS_ROOT_DIR}/proc >/dev/null 2>&1
    [ -d ${SYS_ROOT_DIR}/proc ] || mkdir -p ${SYS_ROOT_DIR}/proc
    mount -n -t proc proc ${SYS_ROOT_DIR}/proc
    if [ $? -ne 0 ]; then
        F_debug_log "Mount /proc failed."
        return 1
    fi

    F_product_config_pre_apply || {
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Pre-apply of Product Data/Config failed."
        F_prompt_log "Product defined pre-check function \"F_product_config_pre_apply\" execute failed. ${ERRMSG_ASK_HELP}."
        ERROR_NUMBER=${ERR_PRODUCT_PRE_CHECK}
        return 1
    }
    F_product_db_pre_apply || {
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Pre-apply of Product DB failed."
        F_prompt_log "Product defined pre-check function \"F_product_db_pre_apply\" execute failed. ${ERRMSG_ASK_HELP}."
        ERROR_NUMBER=${ERR_PRODUCT_PRE_CHECK}
        return 1
    }


    #copy ROOT_PATH to SYS_ROOT_PATH
    #mount -t ext4 "${HIDDEN_PARTS_ARR["new_root_partition"]}" "${SYS_ROOT_DIR}"
    #rm -rf ${SYS_ROOT_DIR}/

    F_debug_log "Move root data to new root partition"
    # Tar root data exclude /proc /sys /dev /run /var/tmp/root2 /opt/TrendMicro /var/local_data    
    /usr/bin/tar cf - / --exclude=/proc --exclude=/sys --exclude=/dev --exclude=/run --exclude=/var/tmp/root2 --exclude=/opt/TrendMicro --exclude=/var/local_data --exclude=/usr/local/openva-update-tools --exclude=/var/log_data --exclude=/var/lib/pgsql | tar xf - -C ${SYS_ROOT_DIR}/
    
    ret=$?
    if [ $ret -ne 0 ]; then
        F_error_log "$FUNCNAME" "`expr $LINENO - 1`" "Could not move root data to new partition, error code:[$ret]."
        cd -
        return 1
    fi
    mkdir ${SYS_ROOT_DIR}/dev ${SYS_ROOT_DIR}/dev/pts ${SYS_ROOT_DIR}/dev/shm/ ${SYS_ROOT_DIR}/sys ${SYS_ROOT_DIR}/var/tmp/root2

    # Create from_openva_upgrade flag to skip DB migration
    # DB migration will be done after reboot the system.
    /usr/bin/touch ${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/from_openva_upgrade.flag

    return 0
}

#Description: 
# Product callback function which will be executed before apply import setting. 
# Some settings are only in the new root partition, but they will be overrided by old settings in F_import_settings().
#
#Arguments:
# None
#
#Side effects:
# None
#
#Returns:
# 0 if successful, others if failed.
F_product_pre_import()
{
    F_debug_log "In product pre-import"

    F_debug_log "Backup service account before import settings"
    grep -E '^elasticsearch:|^rabbitmq:' ${SYS_ROOT_DIR}/etc/shadow > /var/tmp/pre_import_backup_shadow
    grep -E '^elasticsearch:|^rabbitmq:' ${SYS_ROOT_DIR}/etc/passwd > /var/tmp/pre_import_backup_passwd
    grep -E '^elasticsearch:|^rabbitmq:' ${SYS_ROOT_DIR}/etc/group > /var/tmp/pre_import_backup_group
    return 0
}

#Description: 
# Product callback function which will be executed in F_upgrade().
# Because not all rpms will be installed in all server mode, install specific rpms after common rpms are installed.
#
#Arguments:
# None
#
#Side effects:
# None
#
#Returns:
# 0 if successful, others if failed.
F_product_upgrade()
{
    F_debug_log "In product upgrade to install specific rpms."
    local SERVER_MODE="$(cat /opt/TrendMicro/Pixiebob/server_mode)"

    if [ ${SERVER_MODE} == "ALL" ]; then
        rpm --root $SYS_ROOT_DIR -Uvh --force $UNPACK_DIR/packages/productUpdate/*.rpm > ${UPDATE_TOOL_FOLDER}/log/upgrade/rpm_install.log 2>&1
        if [ $? -ne 0 ]; then
            return 1
        fi
    else
        F_debug_log "$SERVER_MODE: skip install product rpms"
    fi

    return 0
}

#Description: 
# Product callback function which will be executed after apply upgrade. 
#
#Arguments:
# None
#
#Side effects:
# None
#
#Returns:
# 0 if successful, others if failed.
F_product_post_apply()
{
    F_debug_log "In product post-apply."
    # This is optional, if there is no action need to do just leave it empty.
    F_add_recovery_func "F_product_post_apply_recovery"

    F_debug_log "Umount /proc"
    umount -f ${SYS_ROOT_DIR}/proc
    if [ $? -ne 0 ]; then
        F_debug_log "Umount /proc failed."
        return 1
    fi

    # Copy AU_data
#    F_debug_log "Copy AU engine and pattern"
#    #rm -rf /var/app_data/AU_data/*
#    [ -d /var/app_data/.AU_data.bak ] && rm -rf /var/app_data/.AU_data.bak
#    mkdir -p /var/app_data/.AU_data.bak
#    mv /var/app_data/AU_data/* /var/app_data/.AU_data.bak/
#    cp -a ${SYS_ROOT_DIR}/var/app_data/AU_data/* /var/app_data/AU_data/

    # Set hostname to ${SYS_ROOT_DIR}/etc/hostname
    F_debug_log "Update /etc/hostname. Ignore"
    #local host_name=`hostname`
    #local host_name_bc=`echo ${host_name} | bc 2>/dev/null`
    #if [ "${host_name}" = "${host_name_bc}" ]; then
    #    F_debug_log "Host name is full numeric, change it to default name:Pixiebob"
    #    local host_name_new="Pixiebob"
    #    echo "${host_name_new}" > ${SYS_ROOT_DIR}/etc/hostname
    #    sed -i "s/^HOSTNAME=.*$/HOSTNAME=${host_name_new}/g" ${SYS_ROOT_DIR}/etc/sysconfig/network
    #    sed -i "s/\s*${host_name}\s*$/ ${host_name_new}/g" ${SYS_ROOT_DIR}/etc/hosts
    #    psql dtasdb dtasuser -c "UPDATE tb_global_setting set value='${host_name_new}' where key='configuration.management_server.hostname';"
    #    if [ $? -ne 0 ]; then
    #        F_debug_log "Failed to change it to default name"
    #        return 1
    #    fi
    #    echo "1" > ${SYS_ROOT_DIR}/opt/hostname_changed
    #    chmod 777 ${SYS_ROOT_DIR}/opt/hostname_changed
    #else
    #    echo "${host_name}" > ${SYS_ROOT_DIR}/etc/hostname
    #fi

    # Export user settings
    F_debug_log "Export user settings. ignore"
    #[ -f /var/app_data/tmp/config_backup.dat ] && rm -f /var/app_data/tmp/config_backup.dat
    #    php /opt/TrendMicro/DTAS/ManagementServer/bin/config_io/config_io.php -a export -m globalsetting -o /var/app_data/tmp/config_backup.dat
    #if [ $? -ne 0 ] || [ ! -f /var/app_data/tmp/config_backup.dat ]; then
    #    F_debug_log "Export user settings failed."
    #    return 1
    #fi

    # Restore system account password to new root system
    F_debug_log "Restore system account's password."
    grep -E '^root:|^admin:' /etc/shadow > /var/tmp/backup_shadow
    sed -i 's#$1$pMorMsd2$rkQuAaWxDKDxNImG5ly2N/#$6$rlBJ2QCNw425wBdG$A7PLC7lyzi9Yy8xgugMB/3E.gU7k9vPDq5qGsCKVijGDAw/MuOuECitawtjllF1kfxAy4TrKXUR.gslIKxulv0#g' /var/tmp/backup_shadow
    sed -i -e '/^root:.*$/d' \
           -e '/^admin:.*$/d' \
           ${SYS_ROOT_DIR}/etc/shadow
    #grep "^hacluster:" /etc/shadow >/dev/null 2>&1
    #if [ $? -eq 0 ]; then
    #    # if hacluster exists, restore it; else, keep it
    #    sed -i '/^hacluster:.*$/d' ${SYS_ROOT_DIR}/etc/shadow
    #fi
    cat /var/tmp/backup_shadow >> ${SYS_ROOT_DIR}/etc/shadow


    # Backup textUI accounts
    F_debug_log "Backup textUI accounts for migration."
    name_list=$(cat /etc/passwd | awk '{FS=":"} $7 == "/opt/TrendMicro/Pixiebob/textUI/admin_shell" {print $1}')
    
    for name in name_list
    do
        if [ "${name}" != "admin" ]; then
            passwd=$(cat /etc/shadow | awk '{FS=":"} $1 == name {print $2}' name="${name}")
            echo "${name}:${passwd}" >> ${SYS_ROOT_DIR}/tmp/textUI_accounts
        fi
    done

    # Restore elasticsearch/rabbitmq acoount to new root system
    # These accounts are backup in F_pre_import_setting
    F_debug_log "Restore service account"
    sed -i -e '/^elasticsearch:.*$/d' \
           -e '/^rabbitmq:.*$/d' \
           ${SYS_ROOT_DIR}/etc/shadow
    sed -i -e '/^elasticsearch:.*$/d' \
           -e '/^rabbitmq:.*$/d' \
           ${SYS_ROOT_DIR}/etc/passwd
    sed -i -e '/^elasticsearch:.*$/d' \
           -e '/^rabbitmq:.*$/d' \
           ${SYS_ROOT_DIR}/etc/group

    cat /var/tmp/pre_import_backup_shadow >> ${SYS_ROOT_DIR}/etc/shadow
    cat /var/tmp/pre_import_backup_passwd >> ${SYS_ROOT_DIR}/etc/passwd
    cat /var/tmp/pre_import_backup_group >> ${SYS_ROOT_DIR}/etc/group

    # Restore system account user id to new root system
    # since the rpm install order may different with anaconda install
    F_debug_log "Restore system account's uid."
    grep -E '^polkitd:|^libstoragemgmt:|^chrony:' /etc/passwd > /var/tmp/backup_passwd
    grep '^polkitd:' /etc/passwd >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^polkitd:.*$/d' ${SYS_ROOT_DIR}/etc/passwd
    fi
    grep '^libstoragemgmt:' /etc/passwd >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^libstoragemgmt:.*$/d' ${SYS_ROOT_DIR}/etc/passwd
    fi
    grep '^chrony:' /etc/passwd >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^chrony:.*$/d' ${SYS_ROOT_DIR}/etc/passwd
    fi

    cat /var/tmp/backup_passwd >> ${SYS_ROOT_DIR}/etc/passwd
    reload_homedir ${SYS_ROOT_DIR}/etc/passwd "libstoragemgmt" ${SYS_ROOT_DIR}
    reload_homedir ${SYS_ROOT_DIR}/etc/passwd "chrony" ${SYS_ROOT_DIR}

    # Restore system account group id to new root system
    # since the rpm install order may different with anaconda install
    F_debug_log "Restore system account's gid."
    grep -E '^ssh_keys:|^polkitd:|^libstoragemgmt:|^chrony:' /etc/group > /var/tmp/backup_group
    grep '^ssh_keys:' /etc/group >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^ssh_keys:.*$/d' ${SYS_ROOT_DIR}/etc/group
    fi
    grep '^polkitd:' /etc/group >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^polkitd:.*$/d' ${SYS_ROOT_DIR}/etc/group
    fi
    grep '^libstoragemgmt:' /etc/group >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^libstoragemgmt:.*$/d' ${SYS_ROOT_DIR}/etc/group
    fi
    grep '^chrony:' /etc/group >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        sed -i '/^chrony:.*$/d' ${SYS_ROOT_DIR}/etc/group
    fi
    cat /var/tmp/backup_group >> ${SYS_ROOT_DIR}/etc/group

    # Change UID /etc/pam.d
    sed -i 's/ 1000 / 500 /g' ${SYS_ROOT_DIR}/etc/pam.d/*

    if [ ! -L /etc/localtime ]; then
        # /etc/localtime is not a link, need fix this bug
        F_debug_log "/etc/localtime is not a link, create softlink"
        rm -f ${SYS_ROOT_DIR}/etc/localtime
        ln -fs /usr/share/zoneinfo/UTC ${SYS_ROOT_DIR}/etc/localtime
    fi

    # Set locale to ${SYS_ROOT_DIR}/etc/locale.conf
    if [ ! -f ${SYS_ROOT_DIR}/etc/locale.conf ] && [ "$LANG" != "" ]; then
        F_debug_log "Update /etc/locale.conf."
        echo "LANG=\"$LANG\"" > ${SYS_ROOT_DIR}/etc/locale.conf
    fi

    # Set keyboard to ${SYS_ROOT_DIR}/
    #if [ "$LANG" = "ja_JP.UTF-8" ]; then
    #    F_debug_log "Update keyboard layout."
    #    sudo chroot ${SYS_ROOT_DIR} > /dev/null 2>&1 <<EOF
    #    /usr/bin/localectl set-keymap jp106
#EOF
    #fi

    #old_id=`cat ${SYS_ROOT_DIR}/etc/machine-id`
    #if [ "${old_id}" == "" ]; then
    #    machine_id=`uuidgen | sed 's/-//g'`
    #    F_debug_log "Update /etc/machine-id: ${machine_id}."
    #    if [ "${machine_id}" != "" ]; then
    #        echo "${machine_id}" > ${SYS_ROOT_DIR}/etc/machine-id
    #    fi
    #else
    #    F_debug_log " /etc/machine-id exists."
    #fi

    #local sqlite_path=/opt/TrendMicro/PI/ui/web/htdocs/trend-widget-framework/repository/db/sqlite
    #local new_sqlite_path=/var/app_data/conf/sqlite
    #if [ -d ${sqlite_path} ] && [ ! -L ${sqlite_path} ]; then
    #    F_debug_log "Restore sqlite."
    #rm -rf ${new_sqlite_path}
    #cp -a ${sqlite_path} /var/app_data/conf/
    #fi

    #if [ -f /var/app_data/log/ddan_rrd.db ] && [ ! -f /var/app_data/conf/ddan_rrd.db ]; then
    #    F_debug_log "Change ddan_rrd.db path to /var/app_data/conf."
    #    cp -af /var/app_data/log/ddan_*rrd.db /var/app_data/conf/
    #fi

    # if 5.5 -> above, keep old setting of SSL protocol
    #grep "^5.1" /var/app_data/tmp/is_upgrade >/dev/null 2>&1
    #if [ $? -ne 0 ]; then
    #    local ssl_proto_str=`grep ^SSLProtocol /etc/httpd/conf.d/ssl.conf 2>/dev/null`
    #    if [ "${ssl_proto_str}" != "" ] && [ -f ${SYS_ROOT_DIR}/etc/httpd/conf.d/ssl.conf ]; then
    #        F_debug_log "Restore SSL protocol to $ssl_proto_str."
    #        sed -i "s/^SSLProtocol.*$/${ssl_proto_str}/g" ${SYS_ROOT_DIR}/etc/httpd/conf.d/ssl.conf
    #    fi
    #fi

    # Set 01_user for grub to encrypt
    F_debug_log "Set grub root password."
    local grub_user_conf=${SYS_ROOT_DIR}/etc/grub.d/01_users
    echo "#!/bin/sh -e" > ${grub_user_conf}
    echo "cat << \"EOF\"" >> ${grub_user_conf}
    echo "set superusers=\"root\"" >> ${grub_user_conf}
    echo "export superusers" >> ${grub_user_conf}
    echo "password_pbkdf2 root grub.pbkdf2.sha512.10000.DDAFA162EBF374F9F733FF87A073667AAFA00604D19C9981D7035AE9B6CF4413871C8AD36F09D2BFAEE969FB296E4341E08637FC49905D8EE063D83DD6DD4272.9C498E6A4D4B81647C42B5F6C2880CF6F0B0D44BC6DE58E925D5C44211651A9A353388D2C15798E64CAAE06DBAE824A03D77C3D04F66545156DE9AAC044D307C" >> ${grub_user_conf}
    echo "EOF" >> ${grub_user_conf}
    chmod 755 ${grub_user_conf}

    # Save upgrade log
#    F_debug_log "Save openva upgrade log"
#    rm -rf ${SYS_ROOT_DIR}/usr/local/openva-update-tools/log/*
#    cp -a /usr/local/openva-update-tools/log/* ${SYS_ROOT_DIR}/usr/local/openva-update-tools/log/

    # remove the is_upgrade flag file in SYS_ROOT_PATH, because install_ddan.sh has finished.
#    rm -f ${SYS_ROOT_DIR}/var/app_data/tmp/is_upgrade

    #setup product variable

    F_debug_log "Remove upgrade flag"
    if [ -f "${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/from_openva_upgrade.flag" ]; then
        rm ${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/from_openva_upgrade.flag
    fi

    # Create migrate flag for DB upgrading. 
    F_debug_log "Create migrate flag for DB upgrading"
    /usr/bin/touch ${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/migrate.flag

    # Service up will cause db chema change.
    # Because DB will be upgrading after system reboot, disable pixiebob service to avoid db duplicate.
    # These service will be enable again in init_pixiebob.sh.
    F_debug_log "Disable pixiebob service in new root"
    /usr/bin/rm ${SYS_ROOT_DIR}/etc/systemd/system/multi-user.target.wants/pixiebob-*

    # Post apply config/Data/DB
    F_product_db_post_apply
    F_product_config_post_apply

    # Remove backup folder
    if [ -e "$backup_dst/ddd_backup" ];then
        rm -rf "$backup_dst/ddd_backup"
    fi

    # Update rc.local file
    if [ -f "${SYS_ROOT_DIR}/etc/rc.d/rc.local" ]; then
        enable_hugepage=`grep "transparent_hugepage" ${SYS_ROOT_DIR}/etc/rc.d/rc.local`
        init_pixiebob=`grep "/opt/TrendMicro/Pixiebob" ${SYS_ROOT_DIR}/etc/rc.d/rc.local`
        if [ "$enable_hugepage" == "" ]; then
            echo "echo never > /sys/kernel/mm/transparent_hugepage/enabled" >> ${SYS_ROOT_DIR}/etc/rc.d/rc.local
            echo "echo never > /sys/kernel/mm/transparent_hugepage/defrag" >> ${SYS_ROOT_DIR}/etc/rc.d/rc.local
        fi

        if [ "$init_pixiebob" == "" ]; then
            echo "sh /opt/TrendMicro/Pixiebob/Server_dep/scripts/post_install.sh" >> ${SYS_ROOT_DIR}/etc/rc.local
            echo "sh /opt/TrendMicro/Pixiebob/Server/init/scripts/init_pixiebob.sh" >> ${SYS_ROOT_DIR}/etc/rc.local
        fi
        chmod +x ${SYS_ROOT_DIR}/etc/rc.local
    else
        echo "sh /opt/TrendMicro/Pixiebob/Server_dep/scripts/post_install.sh" > ${SYS_ROOT_DIR}/etc/rc.local
        echo "sh /opt/TrendMicro/Pixiebob/Server/init/scripts/init_pixiebob.sh" >> ${SYS_ROOT_DIR}/etc/rc.local
        echo "echo never > /sys/kernel/mm/transparent_hugepage/enabled" >> ${SYS_ROOT_DIR}/etc/rc.d/rc.local
        echo "echo never > /sys/kernel/mm/transparent_hugepage/defrag" >> ${SYS_ROOT_DIR}/etc/rc.d/rc.local
        chmod +x ${SYS_ROOT_DIR}/etc/rc.local
    fi

    # Backup openva folder and copy debug log to new root
    rm -rf ${SYS_ROOT_DIR}/usr/local/openva-update-tools/    
    mkdir -p ${SYS_ROOT_DIR}/usr/local/openva-update-tools/
    cp -a /usr/local/openva-update-tools/* ${SYS_ROOT_DIR}/usr/local/openva-update-tools/
    # Remove flag files
    rm -f ${SYS_ROOT_DIR}/usr/local/openva-update-tools/.isinprogress
    rm -f ${SYS_ROOT_DIR}/usr/local/openva-update-tools/.updateongoing

    # Backup updatestatus file
    mv -f ${SYS_ROOT_DIR}/usr/local/openva-update-tools/log/updatestatus ${SYS_ROOT_DIR}/usr/local/openva-update-tools/log/updatestatus.`date +"%s"`

    return 0
}

#Description: 
# The corresponding error-handling function for product pre-apply step.
F_product_pre_apply_recovery()
{
    F_debug_log "Product pre-apply recovery."

    #F_debug_log "Remove upgrade flag."
    #rm -f /var/app_data/tmp/is_upgrade
    #rm -f ${SYS_ROOT_DIR}/var/app_data/tmp/is_upgrade

    F_debug_log "Umount /proc"
    umount -f ${SYS_ROOT_DIR}/proc >/dev/null 2>&1

    F_debug_log "Product config pre-apply recovery"
    F_product_config_pre_apply_recovery
    F_debug_log "Product db pre-apply recovery"
    F_product_db_pre_apply_recovery


    if [ ${ldap_moved} -eq 1 ]; then
        F_debug_log "Recover LDAP cert folder"
        # Move LDAP folder
        if [ -d "${new_ldap_path}" ]; then
            mv -f "${new_ldap_path}" "${old_ldap_path}"
        fi
    fi

    #for serv in local_rrd tmfbed notification_daemon resource_monitor va_monitor data-backup cluster_agent crond; do

    db_service=postgresql-9.3.service
    rpm -q postgresql96
    flag_upgrade_to_pgsql96=$?
    if [ ${flag_upgrade_to_pgsql96} -eq 0 ]; then
        db_service=postgresql-9.6.service
    fi

    F_debug_log "Database service:${db_service}"

    for serv in ${db_service} nginx pixiebob-agent pixiebob-portal pixiebob-resource pixiebob-scheduler pixiebob-log-receiver; do
        #get_status data-backup
        if [ $? -eq 0 ]; then # stopped
            F_debug_log "Start $serv."
            set_status $serv start
        else
            F_debug_log "$serv already start, skip start."
        fi
    done

    #if [ -f /var/www/html/web_service/apps/sampleUpload/module/UploadSampleModule.php.org ]; then
    #    F_debug_log "Start web service of product intergration."
    #    [ -f /var/www/html/web_service/apps/sampleUpload/module/UploadSampleModule.php ] && rm -f /var/www/html/web_service/apps/sampleUpload/module/UploadSampleModule.php
    #    /bin/cp -f /var/www/html/web_service/apps/sampleUpload/module/UploadSampleModule.php.org /var/www/html/web_service/apps/sampleUpload/module/UploadSampleModule.php
    #else
    #    F_debug_log "Skip start web service of product intergration."
    #fi

    F_debug_log "Remove upgrade flag"
    if [ -f "${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/from_openva_upgrade.flag" ]; then
        rm ${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/from_openva_upgrade.flag
    fi

    return 0
}

#Description: 
# The corresponding error-handling function for product post-apply step.
F_product_post_apply_recovery()
{
    F_debug_log "Product post-apply recovery."

    # Rollback AU_update
#    F_debug_log "Rollback AU engine and pattern"
#    if [ -d /var/app_data/.AU_data.bak ]; then
#        rm -rf /var/app_data/AU_data/*
#        mv /var/app_data/.AU_data.bak/* /var/app_data/AU_data/
#        rm -rf /var/app_data/.AU_data.bak
#    fi

    F_debug_log "Product config post-apply recovery"
    F_product_config_post_apply_recovery
    F_debug_log "Product db post-apply recovery"
    F_product_db_post_apply_recovery

    local host_name=`hostname`
    local host_name_bc=`echo ${host_name} | bc 2>/dev/null`
    #if [ "${host_name}" = "${host_name_bc}" ]; then
    #    F_debug_log "Restore hostname to tb_global_setting"
    #    psql dtasdb dtasuser -c "UPDATE tb_global_setting set value='${host_name}' where key='configuration.management_server.hostname';"
    #fi

    # Delete export files 
    #F_debug_log "Delete export files"
    #[ -e /var/app_data/tmp/config_backup.dat ] && rm -f /var/app_data/tmp/config_backup.dat

    #F_debug_log "Umount /proc"
    #umount -f ${SYS_ROOT_DIR}/proc >/dev/null 2>&1
    #umount -l ${SYS_ROOT_DIR}/proc >/dev/null 2>&1

    #F_debug_log "Remove migrate flag created by productUpdate.sh"
    #if [ -f "${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/migrate.flag" ]; then
    #    rm ${SYS_ROOT_DIR}/opt/TrendMicro/Pixiebob/Server/init/migrate.flag
    #fi

    #F_debug_log "Product config pre-apply recovery"
    #F_product_config_post_apply_recovery
    #F_debug_log "Product db pre-apply recovery"
    #F_product_db_post_apply_recovery

    return 0
}

. /usr/local/openva-update-tools/common
